import { HttpService } from '@nestjs/axios';
import { HttpException, Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { JwtService } from '@nestjs/jwt';
import * as bcrypt from 'bcrypt';
import { AuthPayloadInterface } from 'src/interfaces/auth_payload_jwt.interface';
import { RecoveryPayloadInterface } from 'src/interfaces/recovery_payload_jwt.interface';
import { RegisterPayloadInterface } from 'src/interfaces/register_payload_jwt.interface';

@Injectable()
export class UtilsService {
    //Array de errores generados durante la peticion
    private http_error_cause: string[] = [];

    constructor(
        private readonly HttpService: HttpService,
        private readonly JwtService: JwtService,
        private readonly ConfigService: ConfigService,
    ) { }

    /**
     * Retorna una contraseña  y su hash apartir de un string
     * @returns
     */
    generatePassword(password: string): { password: string, hash: string } {

        const response: { password: string, hash: string } = { password: '', hash: '' };

        const saltOrRounds = 10;
        const hash = bcrypt.hashSync(password, saltOrRounds);

        response.password = password;
        response.hash = hash;

        return response;
    }

    /**
     * Genera un Status http y finaliza la peticion
     * @param code 
     * @param message 
     * @param cause 
     * @returns 
     */
    response(code: number, message?: string, cause?: any[]): void {

        const httpStatusCodes = {

            // Peticiones correctas
            200: { status: 200, message: 'Correcto' },
            201: { status: 201, message: 'Creado' },

            // Errores del cliente
            400: { status: 400, error: 'Solicitud Incorrecta', message: 'Formato de solicitud no valida' },
            401: { status: 401, error: 'No Autorizado', message: 'Token no valido' },
            404: { status: 404, error: 'No Encontrado', message: 'No se encontro el recurso solicitado' },
            418: { status: 418, error: 'Soy una tetera', message: 'Los becarios estan procesando los datos' },

            // Errores del servidor
            500: { status: 500, error: 'Error Interno del Servidor', message: 'No se logro procesar la solicitud' },
            503: { status: 503, error: 'Servicio No Disponible', message: 'Servicio suspendido' },

        };

        if (!httpStatusCodes[code]) {
            return null;
        }

        const response = httpStatusCodes[code];

        if (message) {
            response.message = message;
        }

        if (cause) {
            response.cause = cause;
        } else {

            const tmp = this.getHttpCause();

            if (tmp.length > 0) {
                response.cause = tmp;
            }

        }


        throw new HttpException(response, code);
    }

    /**
 * Genera un Status http con un contenido especificado
 * @param code 
 * @param message 
 * @returns 
 */
    customResponse(code: number, content?: any): void {

        //Notifica al interceptor de errores, que muestre los datos sin filtrar
        content.custom = true;

        throw new HttpException(content, code);
    }


    /**
     * Añade registro de error al arreglo cause de utils.response()
     * @param cause 
     */
    setHttpCause(cause: string): void {
        this.http_error_cause.push(cause);
    }

    /**
     * Añade registro de error al arreglo cause de utils.response()
     * @param cause 
     */
    getHttpCause(): string[] {
        const tmp = this.http_error_cause;
        this.http_error_cause = [];
        return tmp;
    }

    /**
     * Genera un access_token de autentificacion, se genera apartir de un inicio de sesion
     * @param user_id 
     * @returns 
     */
    generateAuthToken(user_id: number): string {

        const payloadAccessToken: AuthPayloadInterface = { user_id: user_id };

        const key = this.ConfigService.get('AUTH_JWT');
        const expires = this.ConfigService.get('AUTH_JWT_EXPIRES')

        return this.JwtService.sign(payloadAccessToken, { secret: key, expiresIn: expires })
    }

    /**
     * Genera un access_token de autentificacion, se genera apartir de la accion de registrar un usuario
     * @param user_id 
     * @returns 
     */
    generateActivateAccountToken(user_id: number): string {

        const payloadAccessToken = { user_id: user_id };

        const key = this.ConfigService.get('REGISTER_JWT');
        const expires = this.ConfigService.get('REGISTER_JWT_EXPIRES')
        return this.JwtService.sign(payloadAccessToken, { secret: key, expiresIn: expires })
    }

    /**
     * Genera un token de autenticacion, se genera apartir de la accion de recuperar una contraseña
     * @param user_id 
     * @returns 
     */
    generateForgotPasswordToken(user_id: number): string {

        const payloadAccessToken: RecoveryPayloadInterface = { user_id: user_id };

        const key = this.ConfigService.get('RECOVERY_JWT');
        const expires = this.ConfigService.get('RECOVERY_JWT_EXPIRES')

        return this.JwtService.sign(payloadAccessToken, { secret: key, expiresIn: expires })
    }

    /**
    * Valida un token de auntentificacion "access_token" generado en un inicio de sesion
    * @param token 
    * @returns 
    */
    verifyAuthToken(token: string) {
        try {

            const key = this.ConfigService.get('AUTH_JWT');

            const verify = this.JwtService.verify(token, { secret: key });

            return { payload: verify, error: '', expired: false };
        } catch(err) {
            if(err.name === 'TokenExpiredError') {
                return { payload: null, error: 'Token expirado', expired: true };
            } else {
                return { payload: null, error: 'Token no valido', expired: false };
            }
        }
    }

    /**
     * Valida un token de auntentificacion "access_token" generado en un registro de usuario
     * @param token 
     * @returns 
     */
    verifyActivateAccount(token: string) {
        try {
            const key = this.ConfigService.get('REGISTER_JWT');

            const verify = this.JwtService.verify<RegisterPayloadInterface>(token, { secret: key });

            return { payload: verify, error: '', expired: false };
        } catch(err) {
            if(err.name === 'TokenExpiredError') {
                return { payload: null, error: 'Token expirado', expired: true };
            } else {
                return { payload: null, error: 'Token no valido', expired: false };
            }
        }
    }

        /**
     * Valida un token de auntentificacion "access_token" generado en recuperacion de contraseña
     * @param token 
     * @returns 
     */
    verifyForgotPasswordToken(token: string) {
        try {
            const key = this.ConfigService.get('RECOVERY_JWT');
            const verify = this.JwtService.verify<RecoveryPayloadInterface>(token, { secret: key });
            return { payload: verify, error: '', expired: false };
        } catch(err) {
            console.error(err);
            if(err.name === 'TokenExpiredError') {
                return { payload: null, error: 'Token expirado', expired: true };
            } else {
                return { payload: null, error: 'Token no valido', expired: false };
            }
        }

    }

    /**
     * Retorna los accesos FTP del servidor assets
     */
    getFtpAccess() {
        const host = this.ConfigService.get('FTP_HOST');
        const user = this.ConfigService.get('FTP_USER');
        const password = this.ConfigService.get('FTP_PASSWORD')

        return { host, user, password };
    }

}