import { Injectable } from '@nestjs/common';
import { UtilsService } from '../../utils/utils.service';
import { DtoCreateProduct } from './dto/create_product.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { Products } from 'src/entities/products.entity';
import { DataSource, DeepPartial, FindOptionsRelations, FindOptionsSelect, Repository } from 'typeorm';
import * as path from 'path';
import * as fs from 'fs';
import * as sharp from 'sharp';
import { Users } from 'src/entities/users.entity';
import { FtpService } from '../../ftp/ftp.service';
import { ProductsPhotos } from 'src/entities/products_photos.entity';
import { DtoUploadImage } from './dto/upload_image.dto';
import { ConfigService } from '@nestjs/config';
import { DtoQueryFilters } from './dto/query_filters.dto';
import { ProductsCategories } from 'src/entities/products_categories.entity';
import { DtoUpdatePositionImages } from './dto/update_position_images.dto';
import { DtoUpdateProduct } from './dto/update_product.dto';
import { IndexCategories } from 'src/entities/index_categories.entity';
import { CartItems } from 'src/entities/cart_items.entity';
import { Cart } from 'src/entities/cart.entity';

@Injectable()
export class ProductService {
    constructor(
        private readonly UtilsService: UtilsService,
        private readonly FtpService: FtpService,
        private readonly ConfigService: ConfigService,
        @InjectRepository(Products) private readonly ProductsRepository: Repository<Products>,
        @InjectRepository(ProductsPhotos) private readonly ProductsPhotosRepository: Repository<ProductsPhotos>,
        @InjectRepository(ProductsCategories) private readonly ProductsCategoriesRepository: Repository<ProductsCategories>,
        @InjectRepository(IndexCategories) private readonly IndexCategoriesRepository: Repository<IndexCategories>,
        @InjectRepository(CartItems) private readonly CartItemsRepository: Repository<CartItems>,
        @InjectRepository(Cart) private readonly CartRepository: Repository<Cart>,
        private readonly DataSource: DataSource,
    ) { }

    async findAll(filters: DtoQueryFilters) {
        try {

            const { page, limit, search, min_price, max_price, online, category, type_id, order } = filters;

            const query = this.ProductsRepository
                .createQueryBuilder('products')
                .leftJoin('products.photos', 'photos', 'photos.position = :position', { position: 1 })
                .leftJoin('products.type', 'type')
                .leftJoin('products.products_categories', 'products_categories')
                .addSelect(['photos.id', 'photos.small_size_url', 'photos.title'])
                .addSelect(['type.id', 'type.name'])
                .addSelect(['products_categories.category_id'])

            // 🔍 Filtro por nombre de usuario o perfil
            if (search) {
                query.andWhere(
                    '(products.name LIKE :search)',
                    { search: `%${search}%` },
                );
            }

            if (category) {
                const category_tmp = await this.IndexCategoriesRepository.findOne({where: { name: category}})

                if(category_tmp) {
                    query.andWhere(
                        '(products_categories.category_id LIKE :category_id)',
                        { category_id: `${category_tmp.id}` },
                    );
                }
            }

            // 🔍 Filtro por precio
            if (min_price !== undefined) {
                query.andWhere('products.price >= :min_price', { min_price });
            }
            if (max_price !== undefined) {
                query.andWhere('products.price <= :max_price', { max_price });
            }

            // 🔍 Filtro por estado online
            if (online !== undefined) {
                query.andWhere('products.online = :online', { online });
            }

            // 🔍 Filtro por tipo
            if (type_id !== undefined) {
                query.andWhere('products.type_id = :type_id', { type_id });
            }

            if(order) {
                if(order == 'low-to-high') {
                    query.orderBy('products.price', 'ASC');
                } else if(order == 'high-to-low') {
                    query.orderBy('products.price', 'DESC');
                } else {
                    query.orderBy('products.created_at', 'DESC');
                }
            } else {
                query.orderBy('RAND()');
            }

            const [products, total] = await query
                .skip((page - 1) * limit)
                .take(limit)
                .getManyAndCount();

            const STATIC_URL = this.ConfigService.get('STATIC_URL');

            products.forEach(product => {
                if (product.photos.length > 0) {
                    product.photos.sort();
                    product.photos = [product.photos[0]];
                    const photo = product.photos[0];
                    photo.small_size_url = STATIC_URL + photo.small_size_url;
                }
            });

            return {
                products,
                total,
                page: +page,
                last_page: Math.ceil(total / limit),
            }
        } catch (error) {
            console.error(error);
            return {
                products: [],
                total: 0,
                page: 1,
                last_page: 1,
            }
        }
    }

    async findOne(product_id: number, access_token: string) {
        try {

            const select: FindOptionsSelect<Products> = {
                id: true,
                name: true,
                description: true,
                price: true,
                type_id: true,
                photos: {
                    id: true,
                    title: true,
                    small_size_url: true,
                    position: true,
                },
                products_categories: {
                    id: true,
                    category_id: true,
                    categories: {
                        id: true,
                        name: true,
                    },
                },
                type: {
                    id: true,
                    name: true,
                }
            }

            const relations: FindOptionsRelations<Products> = {
                photos: true,
                products_categories: {
                    categories: true,
                },
                type: true,
            }

            const product = await this.ProductsRepository.findOne({where: {id: product_id}, select, relations});

            const STATIC_URL = this.ConfigService.get('STATIC_URL');

            if(product.photos.length > 0){
                product.photos.sort();

                product.photos.forEach(photo => {
                    photo.small_size_url = STATIC_URL + photo.small_size_url;
                });
            }

            let exist_in_cart = false;
            const user = this.UtilsService.verifyAuthToken(access_token.split(' ')[1]);
            if(user.payload) {
                const cart = await this.CartRepository.findOne({where: { user_id: user.payload.id }});
                const car_item = await this.CartItemsRepository.findOne({where: { product_id: product_id, cart_id: cart.id }});
                if(car_item) {
                    exist_in_cart = true;
                }
            }
            return {...product, exist_in_cart};
        } catch (error) {
            return null;
        }
    }

    async create(product: DtoCreateProduct): Promise<Products> {
        const query_runner = this.DataSource.createQueryRunner();
        await query_runner.connect();
        await query_runner.startTransaction();

        try {

            if(product.categories.length == 0) {
                this.UtilsService.setHttpCause('ERR_NOT_CATEGORIES');
                throw new Error;
            }

            const product_created = await query_runner.manager.save(Products, product).catch(err => {
                this.UtilsService.setHttpCause('ERR_CREATE_PRODUCT');
                throw new Error;
            });

            const categories = [];
            const categories_ids = [];
            product.categories.forEach(category => {
                if(!categories_ids.includes(category)) {
                    categories.push({product_id: product_created.id, category_id: category});
                    categories_ids.push(category);
                }
            });

            await query_runner.manager.save(ProductsCategories, categories).catch(err => {
                this.UtilsService.setHttpCause('ERR_CREATE_CATEGORIES');
                throw new Error;
            });

            await query_runner.commitTransaction();
            return product_created;
        } catch (error) {
            console.error(error);
            await query_runner.rollbackTransaction();
            return null;
        } finally {
            await query_runner.release();
        }
    }

    async update(product_id: number, product: DtoUpdateProduct) {
        const query_runner = this.DataSource.createQueryRunner();
        await query_runner.connect();
        await query_runner.startTransaction();

        try {
            console.log(product, ' product');
            if(product.categories.length == 0) {
                this.UtilsService.setHttpCause('ERR_NOT_CATEGORIES');
                throw new Error;
            }

            const product_updated = await query_runner.manager.save(Products, { ...product, id: product_id }).catch(err => {
                this.UtilsService.setHttpCause('ERR_UPDATE_PRODUCT');
                throw new Error;
            });

            const product_categories = await this.ProductsCategoriesRepository.find({where: { product_id: product_id }});

            const delete_category = [];
            const category_exist = [];

            product_categories.forEach(category => {
                if(!product.categories.includes(category.category_id)) {
                    delete_category.push(category.id);
                } else {
                    category_exist.push(category.id);
                }
            });

            if(delete_category.length > 0) {
                await query_runner.manager.delete(ProductsCategories, delete_category).catch((err) => {
                    console.log(err);
                    this.UtilsService.setHttpCause('ERR_DELETE_CATEGORIES');
                    throw new Error;
                });
            }

            const add_category = [];
            product.categories.forEach(category => {
                const exist = product_categories.find(cat => cat.category_id == category);
                if(!exist){
                    add_category.push({
                        product_id,
                        category_id: category,
                    });
                }
            });

            if(add_category.length > 0) {
                await query_runner.manager.save(ProductsCategories, add_category).catch(() => {
                    this.UtilsService.setHttpCause('ERR_UPDATE_PRODUCT_CATEGORIES');
                    throw new Error;
                });
            }

            await query_runner.commitTransaction();
            return product_updated;
        } catch (error) {
            console.log(error);
            await query_runner.rollbackTransaction();
            return null;
        } finally {
            await query_runner.release();
        }
    }

    async uploadImageProduct(file: Express.Multer.File, data: DtoUploadImage, user: Users) {
        try {
            const base_url = '/products';
            const local_path = path.join(`./tmp/${file.filename}`); // /tmp/**.*
            const file_buffer = fs.readFileSync(local_path); // Se lee la imagen que se subió.

            // Se obtiene el nombre hasheado de la imágen
            const parts = file.filename.split('.');
            parts.pop();
            const new_filename = parts[0];

            // Se convierte a WEBP al archivo original
            const webp_buffer_full = await sharp(file_buffer).webp({ quality: 60 }).toBuffer();
            const web_path_full = path.join(`./tmp/f_${new_filename}.webp`); //crea url para subir archivos en local
            fs.writeFileSync(web_path_full, webp_buffer_full); // Sube archivo localmente

            // Se obtiene los datos de la imagen convertida.
            const metadata = await sharp(file_buffer).metadata();

            // SMALL_SIZE_URL
            const width = Math.round(metadata.width / 2);
            const height = Math.round(metadata.height / 2);
            const webp_buffer_small = await sharp(file_buffer).resize(width, height).toBuffer();
            const web_path_small = path.join(`./tmp/s_${new_filename}.webp`); //crea url para subir archivos en local
            fs.writeFileSync(web_path_small, webp_buffer_small); // Sube archivo localmente


            // Carpetas a la que querremos que se escriba dentro del FTP
            const remote_path_without_filename = path.posix.join(base_url, `/${user.id}`, `/${data.product_id}`)
            const remote_path_full = path.posix.join(base_url, `/${user.id}`, `/${data.product_id}`, 'f_' + new_filename + '.webp');
            const remote_path_small = path.posix.join(base_url, `/${user.id}`, `/${data.product_id}`, 's_' + new_filename + '.webp');

            await this.FtpService.connect(this.UtilsService.getFtpAccess());
            await this.FtpService.uploadFile(web_path_full, remote_path_full, remote_path_without_filename);
            await this.FtpService.uploadFile(web_path_small, remote_path_small, remote_path_without_filename);
            await this.FtpService.disconnect();

            fs.unlinkSync(local_path);
            fs.unlinkSync(web_path_small);
            fs.unlinkSync(web_path_full);

            // Guardar registro en BD

            const last_image = await this.ProductsPhotosRepository
            .createQueryBuilder('photos')
            .select('MAX(photos.position)', 'max_position')
            .where('photos.product_id = :product_id', { product_id: data.product_id })
            .getRawOne();
      
            const position_image = last_image.max_position ? last_image.max_position + 1 : 1;

            const photo: DeepPartial<ProductsPhotos> = {
                product_id: data.product_id,
                title: data.title,
                position: position_image,
                small_size_url: '/static' + remote_path_small,
                full_size_url: '/static' + remote_path_full,
            }

            const created = await this.ProductsPhotosRepository.save(photo);

            if (created) {
                return true;
            } else {
                return false;
            }
        } catch (error) {
            console.error(error, "ERR_PRODUCT_UPLOAD_IMAGES");
            if(!data.is_edit) {
                await this.deleteProduct(data.product_id);
            }
            return false;
        }
    }

    async changeStatusProduct(id: number) {
        try {
            
            const exist = await this.ProductsRepository.findOne({where: {id: id}});

            if(!exist) {
                return false;
            }

            const change_status = await this.ProductsRepository.update({id}, {online: exist.online == 1 ? 0 : 1});

            if(change_status.affected > 0) {
                return true;
            } else {
                return false;
            }

            return true;
        } catch (error) {
            return false;
        }
    }

    async updatePositionsImages(product_id: number, data: DtoUpdatePositionImages) {
        try {
            
            if(data.images_position.length == 0){
                this.UtilsService.setHttpCause('ERR_EMPTY_IMAGES');
                return false
            }

            const exist_product = await this.ProductsRepository.findOne({where: {id: product_id}, select: {id: true}});

            if(!exist_product) {
                this.UtilsService.setHttpCause('ERR_PRODUCT_NOT_EXIST');
                return false
            }

            const new_positions = [];

            data.images_position.forEach(image => {
                new_positions.push({
                    id: image.id,
                    position: image.position,
                    product_id
                });
            });

            const response = await this.ProductsPhotosRepository.save(new_positions);

            if(response.length > 0) {
                return true;
            } else {
                return false
            }
        } catch (error) {
            return false
        }
    }

    async deleteImage(product_id: number, image_id: number) {
        try {
            
            const exist = await this.ProductsRepository.findOne({where: {id: product_id}, select: {id: true}});

            if(!exist) {
                this.UtilsService.setHttpCause('ERR_PRODUCT_NOT_EXIST');
                return false;
            }

            const deleted = await this.ProductsPhotosRepository.delete({id: image_id, product_id: exist.id});

            const images = await this.ProductsPhotosRepository.find({where: {product_id}, order: {position: 'ASC'}});

            const new_positions = [];

            images.forEach((image, i) => {
                new_positions.push({
                    id: image.id,
                    position: i + 1
                });
            })


            await this.updatePositionsImages(product_id, { images_position: new_positions });

            if(deleted.affected > 0) {
                return true;
            } else {
                this.UtilsService.setHttpCause('ERR_NOT_AFFECTED');
                return false;
            }

        } catch (error) {
            console.error(error);
            return false;
        }
    }

    private async deleteProduct(product_id: number) {
        try {
            const deleted = await this.ProductsRepository.delete({id: product_id});

            if(deleted.affected > 0) {
                return true;
            } else {
                return false;
            }
        } catch (error) {
            return false;
        }
    }

}
