import { Body, Controller, Delete, Get, Param, ParseIntPipe, Post, Put, Query, Request, UploadedFile, UseGuards, UseInterceptors } from '@nestjs/common';
import { ProductService } from './product.service';
import { UserAdminAuthorizationGuard } from 'src/guards/auth_admin/user_admin_authorization.guard';
import { DtoCreateProduct } from './dto/create_product.dto';
import { UtilsService } from '../../utils/utils.service';
import { FileInterceptor } from '@nestjs/platform-express';
import { FileSizeValidationPipe } from './pipes/fileSizeValidation.pipe';
import { UserAuthorization } from 'src/decorators/user_authorization.decorator';
import { Users } from 'src/entities/users.entity';
import { DtoUploadImage } from './dto/upload_image.dto';
import { DtoQueryFilters } from './dto/query_filters.dto';
import { DtoUpdatePositionImages } from './dto/update_position_images.dto';
import { DtoUpdateProduct } from './dto/update_product.dto';

@Controller('product')
export class ProductController {
  constructor(
    private readonly ProductService: ProductService,
    private readonly UtilsService: UtilsService,
  ) {}

  @Get()
  async findAll(@Query() filters: DtoQueryFilters) {
    return await this.ProductService.findAll(filters);
  }

  @Get('/:product_id')
  async findOne(@Param('product_id') product_id: number, @Request() req: Request) {
    return await this.ProductService.findOne(product_id, req.headers['authorization']);
  }

  @UseGuards(UserAdminAuthorizationGuard)
  @Post()
  async createProduct(@Body() product: DtoCreateProduct) {
    const response = await this.ProductService.create(product)

    if(response){
      return response;
    } else {
      this.UtilsService.response(200);
    }

  }

  @UseGuards(UserAdminAuthorizationGuard)
  @Put('/:id')
  async updateProduct(@Param('id', ParseIntPipe) id: number, @Body() product: DtoUpdateProduct) {
    const response = await this.ProductService.update(id, product);

    if(response) {
      this.UtilsService.response(200);
    } else {
      this.UtilsService.response(400);
    }
  }
  
  @UseGuards(UserAdminAuthorizationGuard)
  @Put('status/:id')
  async changeStatusProduct(@Param('id', ParseIntPipe) id: number) {
    const response = await this.ProductService.changeStatusProduct(id);

    if(response) {
      this.UtilsService.response(200);
    } else {
      this.UtilsService.response(400);
    }

  }

  @UseGuards(UserAdminAuthorizationGuard)
  @Post('/upload')
  @UseInterceptors(FileInterceptor('file'))
  async uploadImagesProduct(@Body() data: DtoUploadImage, @UploadedFile() file: Express.Multer.File, @UserAuthorization() user: Users) {
    
    if(!file || !file.path) {
      this.UtilsService.response(400);
    }

    const uploaded = await this.ProductService.uploadImageProduct(file, data, user);

    if(uploaded) {
      this.UtilsService.response(201);
    } else {
      this.UtilsService.response(400);
    }
  }

  @UseGuards(UserAdminAuthorizationGuard)
  @Put('/position/:id')
  async changeCurrentPositionImages(@Param('id', ParseIntPipe) product_id: number, @Body() data: DtoUpdatePositionImages) {

    const response = await this.ProductService.updatePositionsImages(product_id, data);

    if(response){
      this.UtilsService.response(200);
    } else {
      this.UtilsService.response(400); 
    }
  }

  @UseGuards(UserAdminAuthorizationGuard)
  @Delete('/images/:product_id/:image_id')
  async deleteImage(@Param('product_id', ParseIntPipe) product_id: number, @Param('image_id', ParseIntPipe) image_id: number ) {
    const response = await this.ProductService.deleteImage(product_id, image_id);

    if(response) {
      this.UtilsService.response(200);
    } else {
      this.UtilsService.response(400);
    }

  }

}
