import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Cart } from 'src/entities/cart.entity';
import { Products } from 'src/entities/products.entity';
import { Users } from 'src/entities/users.entity';
import { Repository } from 'typeorm';
import { DtoAddToCart } from './dto/add_to_cart.dto';
import { CartItems } from 'src/entities/cart_items.entity';
import { UtilsService } from '../../utils/utils.service';
import { ConfigService } from '@nestjs/config';

@Injectable()
export class CartService {

    constructor(
        @InjectRepository(Cart) private readonly CartRepository: Repository<Cart>,
        @InjectRepository(Products) private readonly ProductRepository: Repository<Products>,
        @InjectRepository(CartItems) private readonly CartItemsRepository: Repository<CartItems>,
        private readonly UtilsService: UtilsService,
        private readonly ConfigService: ConfigService,
    ) { }

    /**
     * Obtiene el carrito actual del usuario
     * @param user 
     * @returns 
     */
    async getCart(user: Users) {
        try {
            let shopping_cart: Cart = null;

            shopping_cart = await this.CartRepository.findOne(
                { where: { user_id: user.id, active: 1 }, relations: ['items', 'items.product', "items.product.photos"], });

            // En caso de que no tenga ninguno, lo crea
            if (shopping_cart == null) {
                shopping_cart = await this.CartRepository.save({ user_id: user.id, active: 1 })
            }

            // Formatear carrito
            const cart = {
                id: shopping_cart.id,
                price: 0,
                quantity: 0,
                discount: 0,
                items: [],
            }

            if(shopping_cart.items.length > 0) {
                shopping_cart.items.forEach(item => {
                    cart.price += +item.product.price;
                    cart.quantity += item.quantity;

                    const photos = [];

                    if(item.product.photos.length > 0) {
                        const photo = item.product.photos.sort()[0];
                        photos.push({
                            id: photo.id,
                            title: photo.title,
                            small_size_url: this.ConfigService.get('STATIC_URL') + photo.small_size_url,
                            full_size_url: this.ConfigService.get('STATIC_URL') + photo.full_size_url
                        });
                    }

                    cart.items.push({
                        id: item.product_id,
                        item_id: item.id,
                        name: item.product.name,
                        price: item.product.price,
                        quantity: item.quantity,
                        photos: photos
                    });
                });
            }

            return cart;

        } catch (error) {
            console.log(error, ' error');
            return null;
        }
    }

    /**
     * Agrega un producto al carrito
     * @param user 
     * @param data 
     * @returns 
     */
    async addToCart(user: Users, data: DtoAddToCart) {
        try {
            const cart = await this.getCart(user);

            if (!cart) {
                return null;
            }

            const product = await this.ProductRepository.findOne({ where: { id: data.product_id } });

            if (!product) {
                return null;
            }

            let cart_item = await this.CartItemsRepository.findOne({
                where: { cart_id: cart.id, product_id: product.id, },
            });

            if (cart_item) {
                if (product.type_id == 2) {
                    cart_item.quantity += data.quantity;
                } else {
                    this.UtilsService.setHttpCause('PRODUCT_ADDED');
                    return null;
                }
            } else {
                cart_item = this.CartItemsRepository.create({ cart_id: cart.id, product_id: data.product_id, quantity: data.quantity });
            }

            const item = await this.CartItemsRepository.save(cart_item);

            return item;
        } catch (error) {
            return null;
        }
    }

    /**
     * Elimina un producto del carrito
     * @param user 
     * @param item_id 
     * @returns 
     */
    async removeFromCart(user: Users, item_id: number) {
        try {

            const cart = await this.getCart(user);

            if (!cart) {
                this.UtilsService.setHttpCause('ERR_CART_NOT_EXIST');
                return null;
            }

            const cart_item = await this.CartItemsRepository.findOne({ where: { id: item_id } });

            if (!cart_item) {
                this.UtilsService.setHttpCause('ERR_CART_ITEM_NOT_EXIST');
                return null;
            }

            const removed = await this.CartItemsRepository.delete(cart_item);

            if (removed.affected > 0) {
                return await this.getCart(user);
            } else {
                this.UtilsService.setHttpCause('ERR_CART_ITEM_NOT_REMOVED');
                return null;
            }

        } catch (error) {
            this.UtilsService.setHttpCause('ERR_REMOVED_CART_ITEM');
            return null;
        }
    }

    /**
     * Obtiene el total de productos que hay en el carrito
     * @param user 
     * @returns 
     */
    async getTotalProductsCart(user: Users) {
        try {
            const cart = await this.getCart(user);
            return {total: cart.items.length};
        } catch (error) {
            return { total: 0 };
        }
    }
}
