import { Body, Controller, Delete, Get, Param, ParseIntPipe, Post, UseGuards } from '@nestjs/common';
import { CartService } from './cart.service';
import { UserAuthorizationGuard } from 'src/guards/auth/user_authorization.guard';
import { UserAuthorization } from 'src/decorators/user_authorization.decorator';
import { Users } from 'src/entities/users.entity';
import { DtoAddToCart } from './dto/add_to_cart.dto';
import { UtilsService } from '../../utils/utils.service';

@Controller('cart')
export class CartController {
  
  constructor(private readonly CartService: CartService, private readonly UtilsService: UtilsService) {}

  @UseGuards(UserAuthorizationGuard)
  @Get()
  async getCart(@UserAuthorization() user: Users) {
    return this.CartService.getCart(user);
  }

  @UseGuards(UserAuthorizationGuard)
  @Post('/cart_item/add')
  async addToCart(@UserAuthorization() user: Users, @Body() data: DtoAddToCart) {
    const added = await this.CartService.addToCart(user, data);

    if(added) {
      return added;
    } else {
      this.UtilsService.response(400);
    }
  }

  @UseGuards(UserAuthorizationGuard)
  @Delete('/cart_item/:item_id')
  async removeFromCart(@UserAuthorization() user: Users, @Param('item_id', ParseIntPipe) item_id: number) {
    const removed = await this.CartService.removeFromCart(user, item_id);

    if(removed) {
      return removed;
    } else {
      this.UtilsService.response(400);
    }
  }

  @UseGuards(UserAuthorizationGuard)
  @Get('/total_items')
  async getTotalProductsInCart(@UserAuthorization() user: Users) {
    return await this.CartService.getTotalProductsCart(user);
  }

}
