import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { DtoCreateAdminUser } from 'src/auth/dto/create_admin_user.dto';
import { DtoLoginUserAdmin } from 'src/auth/dto/login_user_admin.dto';
import { AdminUsers } from 'src/entities/admin_users.entity';
import { UtilsService } from 'src/utils/utils.service';
import { DeepPartial, FindOptionsSelect, Repository } from 'typeorm';
import * as bcrypt from 'bcrypt';
import { ConfigService } from '@nestjs/config';
import { JwtService } from '@nestjs/jwt';


@Injectable()
export class AuthAdminService {

    constructor(
        private readonly UtilsService: UtilsService,
        private readonly ConfigService: ConfigService,
        private readonly JwtService: JwtService,
        @InjectRepository(AdminUsers) private readonly AdminUsersRepository: Repository<AdminUsers>,


    ){}

    async createAdminUser(data: DtoCreateAdminUser) {

        const exist = await this.AdminUsersRepository.findOne({ where: { email: data.email } });

        if (exist) {
            this.UtilsService.response(401, 'Email existente');
        }

        const password_hash = this.UtilsService.generatePassword(data.password);

        const admin: DeepPartial<AdminUsers> = {
            ...data,
            password: password_hash.hash,
            role_id: data.role_id ? data.role_id : 2
        };

        const created_user = await this.AdminUsersRepository.save(admin);

        if (created_user) {
            this.UtilsService.response(201);
        } else {
            this.UtilsService.response(400);
        }
    }

    async loginAdmin(data: DtoLoginUserAdmin) {

        const select: FindOptionsSelect<AdminUsers> = {
            email: true,
            password: true,
        }

        const user = await this.AdminUsersRepository.findOne({ where: { email: data.email }, select });

        if (!user) {
            this.UtilsService.response(400, 'Usuario no existente');
        }

        const match = await bcrypt.compare(data.password, user.password);

        if (!match) {
            this.UtilsService.response(400, 'Contraseña incorrecta');
        }

        const payload_access_token = { username: user.name, user_id: user.id };
        const payload_refresh_token = { username: user.name, user_id: user.id };

        let expires = this.ConfigService.get('AUTH_ADMIN_JWT_EXPIRES');

        expires = expires.replace('s', '')

        return {
            access_token: this.JwtService.sign(payload_access_token),
            refresh_token: this.JwtService.sign(payload_refresh_token, { secret: user.password, expiresIn: '864000s' }),
        };

    }

    async generateRefreshAccessToken(refresh_token: string) {
        try {

            const payload = this.JwtService.decode(refresh_token, { complete: true });

            const user = await this.AdminUsersRepository.findOne({ where: { id: payload.id }, select: { password: true, id: true, email: true } });

            if(!user) {
                this.UtilsService.response(400);
            }

            const secret = this.ConfigService.get('AUTH_ADMIN_JWT');

            const is_valid = this.JwtService.verify(refresh_token, { secret, algorithms: ['HS384'] });

            if(is_valid) {

                const secret = this.ConfigService.get('AUTH_ADMIN_JWT');
                const expires = this.ConfigService.get('AUTH_ADMIN_JWT_EXPIRES');
        
                const access_token = this.JwtService.sign({ id: user.id, email: user.email }, { secret, expiresIn: expires });
                const refresh_token = this.JwtService.sign({ id: user.id, email: user.email }, { secret, expiresIn: "864000s" });
        
                return { access_token, refresh_token };

            } else {
                this.UtilsService.response(400, 'Token invalido');
            }

        } catch (error) {
            console.error(error);
            this.UtilsService.response(400, 'Token invalido');
        }
    }
}
