import { Injectable, UnauthorizedException } from '@nestjs/common';
import { DtoCreateUser } from './dto/create_user.dto';
import { UtilsService } from '../utils/utils.service';
import { Users } from '../entities/users.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { FindOptionsSelect, Repository } from 'typeorm';
import { ConfigService } from '@nestjs/config';
import { JwtService } from '@nestjs/jwt';
import { DtoLoginUser } from './dto/login_user.dto';
import * as bcrypt from 'bcrypt';
import { MailService } from '../mail/mail.service';
import { DtoForgotPassword } from './dto/forgot_password.dto';
import { DtoChangePassword } from './dto/change_password.dto';
import { AdminUsers } from 'src/entities/admin_users.entity';
import * as jwt from 'jsonwebtoken'
import { DtoUpdateInforUser } from './dto/update_info_user.dto';
import { CartService } from '../modules/cart/cart.service';

@Injectable()
export class AuthService {

    constructor(
        private readonly UtilsService: UtilsService,
        private readonly ConfigService: ConfigService,
        private readonly JwtService: JwtService,
        private readonly MailService: MailService,
        private readonly CartService: CartService,
        @InjectRepository(Users) private readonly UsersRepository: Repository<Users>,
        @InjectRepository(AdminUsers) private readonly AdminUsersRepository: Repository<AdminUsers>,
    ) { }

    /**
   * Retorna access_token generado con el correo y user_id del usuario
   * @param user 
   * @returns 
   */
    async login(user: DtoLoginUser): Promise<{ access_token: string, refresh_token: string }> {

        const select: FindOptionsSelect<Users> = {
            id: true,
            email: true,
            password: true,
            is_active: true,
        }

        const exist_user = await this.UsersRepository.findOne({ where: { email: user.email }, select });

        if (!exist_user) {
            throw new UnauthorizedException('Correo electrónico incorrecto.');
        }

        if (exist_user.is_active == 0) {
            throw new UnauthorizedException('Cuenta no verificada.');
        }

        const passowrd_match = await bcrypt.compare(user.password, exist_user.password)

        if (!passowrd_match) {
            throw new UnauthorizedException('Contraseña incorrecta.');
        }

        const secret = this.ConfigService.get('AUTH_JWT');
        const expires = this.ConfigService.get('AUTH_JWT_EXPIRES');

        const access_token = this.JwtService.sign({ id: exist_user.id, email: exist_user.email }, { secret, expiresIn: expires });
        const refresh_token = this.JwtService.sign({ id: exist_user.id, email: exist_user.email }, { secret, expiresIn: "864000s" });

        return { access_token, refresh_token };
    }

    /**
     * Registrar un usuario
     * @param user 
     * @returns 
     */
    async registerUser(user: DtoCreateUser) {
        try {
            const exist_user = await this.UsersRepository.findOne({ where: { email: user.email } });

            if (exist_user) {
                this.UtilsService.setHttpCause('EMAIL_EXIST');
                return null;
            }

            // Se hashea la contraseña
            const password_hashed = this.UtilsService.generatePassword(user.password);

            // Se crea nuevo usuario
            const new_user = await this.UsersRepository.save({ ...user, password: password_hashed.hash });

            // Se genera token de activación
            const token_activation = this.UtilsService.generateActivateAccountToken(new_user.id);
            console.log(token_activation, ' token_activación');
            // Se envía email
            const sent = await this.MailService.activateAccount({ name: user.name, to: user.email, link_activation: `/activate-account/${token_activation}` });

            if (!sent) {
                // Eliminar usuario
                await this.UsersRepository.delete(new_user.id);
                return false;
            }

            return true;
        } catch (error) {
            console.error(error, ' error');
            return false;
        }
    }

    async me(user: Users) {
        try {

            const { total } = await this.CartService.getTotalProductsCart(user);

            return {...user, products_in_cart: total};
        } catch (error) {
            return null;
        }
    }

    /**
     * Activa la cuenta del usuario
     * @param token 
     * @returns 
     */
    async activateAccount(token: string) {

        if (!token) {
            this.UtilsService.setHttpCause('INVALID_TOKEN');
            this.UtilsService.response(400, 'Token inválido');
        }

        const payload = this.UtilsService.verifyActivateAccount(token);

        if (payload.expired) {
            this.UtilsService.response(400, payload.error)
        }

        if (!payload.payload) {
            this.UtilsService.response(400, payload.error);
        }

        const exist = await this.UsersRepository.findOne({ where: { id: payload.payload.user_id }, select: { id: true, name: true, email: true, is_active: true } });

        if (!exist) {
            this.UtilsService.setHttpCause('USER_NOT_EXIST');
            this.UtilsService.response(404, 'El usuario no existe');
        }

        if (exist.is_active == 1) {
            this.UtilsService.setHttpCause('ACCOUNT_VERIFIED');
            this.UtilsService.response(403, 'La cuenta ya ha sido verificada');
        }

        await this.UsersRepository.update({ id: exist.id }, { is_active: 1 });

        return true;
    }


    async forgotPassword(data: DtoForgotPassword) {
        try {

            const user = await this.UsersRepository.findOne({ where: { email: data.email } });

            if (!user) {
                this.UtilsService.setHttpCause('USER_NOT_EXIST');
                return false;
            }

            const token = this.UtilsService.generateForgotPasswordToken(user.id);

            // Se envía email
            const sent = await this.MailService.forgotPassword({ name: user.name, to: user.email, link_activation: `/auth/change_password/${token}` });

            if (!sent) {
                return false;
            }

        } catch (error) {
            console.error(error);
            return false;
        }
    }

    async changePassword(data: DtoChangePassword) {

        const payload = this.UtilsService.verifyForgotPasswordToken(data.token);

        if (payload.expired) {
            this.UtilsService.response(400, payload.error)
        }

        if (!payload.payload) {
            this.UtilsService.response(400, payload.error);
        }

        const new_password = this.UtilsService.generatePassword(data.password);

        const updated = await this.UsersRepository.update({ id: payload.payload.user_id }, { password: new_password.hash }).catch(err => {
            this.UtilsService.response(500);
            return null;
        });

        const user = await this.UsersRepository.findOne({ where: { id: payload.payload.user_id } });

        await this.MailService.successPasswordChanged({ to: user.email, name: user.name });

        if (updated.affected > 0) {
            this.UtilsService.response(200);
        } else {
            this.UtilsService.response(400);
        }
    }

    async generateRefreshAccessToken(refresh_token: string) {
        try {

            const payload = this.JwtService.decode(refresh_token, { complete: true });

            const user = await this.UsersRepository.findOne({ where: { id: payload.id }, select: { password: true, id: true, email: true } });

            if (!user) {
                this.UtilsService.response(400);
            }

            const secret = this.ConfigService.get('AUTH_JWT');

            const is_valid = this.JwtService.verify(refresh_token, { secret, algorithms: ['HS384'] });

            if (is_valid) {

                const secret = this.ConfigService.get('AUTH_JWT');
                const expires = this.ConfigService.get('AUTH_JWT_EXPIRES');

                const access_token = this.JwtService.sign({ id: user.id, email: user.email }, { secret, expiresIn: expires });
                const refresh_token = this.JwtService.sign({ id: user.id, email: user.email }, { secret, expiresIn: "864000s" });

                return { access_token, refresh_token };

            } else {
                this.UtilsService.response(400, 'Token invalido');
            }

        } catch (error) {
            console.error(error, 'error');
            this.UtilsService.response(400, 'Token invalido');
        }
    }

    async sendEmailActivation(token: string) {

        if (!token) {
            this.UtilsService.setHttpCause('INVALID_TOKEN');
            this.UtilsService.response(400, 'Token inválido');
        }

        const payload = jwt.decode(token);

        const user = await this.UsersRepository.findOne({ where: { id: payload['user_id'] }, select: { id: true, name: true, email: true, is_active: true } });
        
        // Se genera token de activación
        const token_activation = this.UtilsService.generateActivateAccountToken(user.id);

        // Se envía email
        const sent = await this.MailService.activateAccount({ name: user.name, to: user.email, link_activation: `/activate-account/${token_activation}` });

        if(sent) {
            this.UtilsService.response(200);
        } else {
            this.UtilsService.response(400);
        }
    }

    async updateInfoUser(id: number, data: DtoUpdateInforUser) {
        try {
            const exist = await this.UsersRepository.findOne({where: {id}});

            if(!exist) {
                return null;
            }

            const updated = await this.UsersRepository.save({id, ...data});

            if(updated) {
                return updated;
            } else {
                return null;
            }


        } catch (error) {
            return null;
        }
    }
}
