import { Body, Controller, Get, Param, ParseIntPipe, Post, Put, UseGuards } from '@nestjs/common';
import { AuthService } from './auth.service';
import { DtoCreateUser } from './dto/create_user.dto';
import { UtilsService } from '../utils/utils.service';
import { DtoLoginUser } from './dto/login_user.dto';
import { Users } from 'src/entities/users.entity';
import { DtoForgotPassword } from './dto/forgot_password.dto';
import { DtoChangePassword } from './dto/change_password.dto';
import { UserAuthorizationGuard } from 'src/guards/auth/user_authorization.guard';
import { UserAuthorization } from 'src/decorators/user_authorization.decorator';
import { DtoUpdateInforUser } from './dto/update_info_user.dto';

@Controller('auth')
export class AuthController {
  constructor(private readonly AuthService: AuthService, private readonly UtilsService: UtilsService) {}

  @Post('/register')
  async register(@Body() user: DtoCreateUser) {

    const created = await this.AuthService.registerUser(user)
    
    if(created) {
      this.UtilsService.response(201);
    } else {
      this.UtilsService.response(400);
    }
  }

  @Post('/login')
  async login(@Body() user: DtoLoginUser) {
    return await this.AuthService.login(user);
  }

  @UseGuards(UserAuthorizationGuard)
  @Get('/me')
  async getMe(@UserAuthorization() user: Users) {
    const response = await this.AuthService.me(user);

    if(response) {
      return response;
    } else {
      this.UtilsService.response(401);
    }

  }

  @UseGuards(UserAuthorizationGuard)
  @Put('/user')
  async updateInfoUser(@UserAuthorization() user: Users, @Body() data: DtoUpdateInforUser) {
    const response = await this.AuthService.updateInfoUser(user.id, data);

    if(response) {
      return response;
    } else {
      this.UtilsService.response(400);
    }
  }

  @Get('/activate_account/:token')
  async activateAccount(@Param('token') token: string) {
    const activated = await this.AuthService.activateAccount(token);

    if(activated) {
      this.UtilsService.response(200);
    } else {
      this.UtilsService.response(400);
    }

  }

  /**
   * Envía token para poder cambiar la contraseña
   * @param data 
   */
  @Post('/forgot_password')
  async forgotPassword(@Body() data: DtoForgotPassword) {
    const token = await this.AuthService.forgotPassword(data);

    if(token) {
      this.UtilsService.response(200);
    } else {
      this.UtilsService.response(400);
    }

  }

  @Put('/forgot_password')
  async changePassword(@Body() data: DtoChangePassword) {
    return await this.AuthService.changePassword(data);
  }

  @Post('/token')
  async refreshAccessToken(@Body('refresh_token') refresh_token: string) {
    return await this.AuthService.generateRefreshAccessToken(refresh_token);
  }

  @Get('/send_email_activation/:token')
  async resendEmailActivation(@Param('token') token: string) {
    return await this.AuthService.sendEmailActivation(token);    
  }
}
